/*:
 * @target MZ
 * @plugindesc [Add-on] HS_ButtonPictureRealtime: バインド永続化＆ロード直後の自動再適用 v1.1.0
 * @author HS/ChatGPT
 *
 * @param defaultBinds
 * @text 初期バインド (pictureId:commonEventId)
 * @type string
 * @default
 * @desc 例: 10:3,11:4 （ピクチャ10→コモン3）/ 既存セーブにも即反映したい場合に設定
 *
 * @help
 * ■何が直るか
 * - ゲームを一度終了→再起動→ロードした直後だけ、特定のピクチャボタンが反応しない
 * - 数秒〜十数秒待つ/メッセージを進めると反応する
 *
 * ■原因(典型例)
 * HS_ButtonPictureRealtime.js は、プラグインコマンド bind により
 * Game_Picture に `_hs_rt_ceId` を付与してクリック判定します。
 * ところが環境によっては「ロード直後〜UI初期化完了まで」の間、
 * ピクチャの再生成/差し替え等でこの一時プロパティが消えることがあり、
 * その時間だけクリック判定がスキップされます。
 *
 * ■解決策
 * 1) bind/unbind の対応を $gameSystem に保存(永続化)
 * 2) ロード直後/ピクチャ再表示(showPicture)のタイミングで自動再適用
 *
 * ■導入手順
 * - 本プラグインは HS_ButtonPictureRealtime.js より下に配置してください。
 * - 既存セーブにも即反映したい場合は、プラグインパラメータ defaultBinds を設定してください。
 *   (設定しない場合でも、今後 bind を実行した後にセーブすれば次回から安定します)
 *
 * ■注意
 * - HS_ButtonPictureRealtime のプラグイン名は固定で「HS_ButtonPictureRealtime」を前提にフックします。
 *   (元プラグインの const PN が同名であることが前提)
 */

(() => {
  'use strict';

  const TARGET_PLUGIN = 'HS_ButtonPictureRealtime';
  const STORE_KEY = '_hsRtPicBindsV1';

  const thisPluginName = (() => {
    try {
      const src = document.currentScript && document.currentScript.src;
      if (!src) return 'HS_ButtonPictureRealtime_PersistBind';
      const file = decodeURIComponent(src.split('/').pop() || '');
      return file.replace(/\.js$/i, '') || 'HS_ButtonPictureRealtime_PersistBind';
    } catch (_) {
      return 'HS_ButtonPictureRealtime_PersistBind';
    }
  })();

  const params = PluginManager.parameters(thisPluginName);
  const defaultBindsText = String(params.defaultBinds || '').trim();

  function parseDefaultBinds(text) {
    // "10:3,11:4" -> {10:3, 11:4}
    const map = Object.create(null);
    if (!text) return map;
    const parts = text.split(',');
    for (const raw of parts) {
      const s = String(raw || '').trim();
      if (!s) continue;
      const m = s.match(/^(\d+)\s*:\s*(\d+)$/);
      if (!m) continue;
      const pid = Number(m[1]);
      const ceid = Number(m[2]);
      if (pid > 0 && ceid > 0) map[pid] = ceid;
    }
    return map;
  }

  const DEFAULT_BINDS = parseDefaultBinds(defaultBindsText);

  function ensureStore() {
    if (typeof $gameSystem === 'undefined' || !$gameSystem) return null;
    if (!$gameSystem[STORE_KEY] || typeof $gameSystem[STORE_KEY] !== 'object') {
      $gameSystem[STORE_KEY] = Object.create(null);
    }
    const store = $gameSystem[STORE_KEY];

    // Seed defaults for missing keys (idempotent)
    for (const k of Object.keys(DEFAULT_BINDS)) {
      const pid = Number(k);
      if (!store[pid]) store[pid] = DEFAULT_BINDS[pid];
    }
    return store;
  }

  function applyBind(pictureId, commonEventId) {
    if (typeof $gameScreen === 'undefined' || !$gameScreen) return;
    const pic = $gameScreen.picture(pictureId);
    if (!pic) return;
    pic._hs_rt_ceId = commonEventId;
    pic._hs_rt_last = 0;
  }

  function applyAllExistingPictures() {
    const store = ensureStore();
    if (!store) return;
    for (const k of Object.keys(store)) {
      const pid = Number(k);
      const ceid = Number(store[pid] || 0);
      if (pid > 0 && ceid > 0) applyBind(pid, ceid);
    }
  }

  // 1) bind/unbind をフックして永続化
  const _callCommand = PluginManager.callCommand;
  PluginManager.callCommand = function(self, pluginName, commandName, args) {
    if (pluginName === TARGET_PLUGIN) {
      const store = ensureStore();
      const cmd = String(commandName || '');
      if (store && (cmd === 'bind' || cmd === 'unbind')) {
        const pid = Number(args && args.pictureId);
        if (pid > 0) {
          if (cmd === 'bind') {
            const ceid = Number(args && args.commonEventId);
            if (ceid > 0) store[pid] = ceid;
          } else {
            delete store[pid];
          }
        }
      }
    }
    return _callCommand.apply(this, arguments);
  };

  // 2) ロード直後に再適用
  const _DataManager_createGameObjects = DataManager.createGameObjects;
  DataManager.createGameObjects = function() {
    _DataManager_createGameObjects.call(this);
    ensureStore();
  };

  const _DataManager_extractSaveContents = DataManager.extractSaveContents;
  DataManager.extractSaveContents = function(contents) {
    _DataManager_extractSaveContents.call(this, contents);
    ensureStore();
    // この時点では spriteset が未生成の場合もあるので、
    // Game_Picture 側だけ先に補強しておく
    applyAllExistingPictures();
  };

  // 3) ピクチャが再表示されたら即再適用 (ピクチャ再生成/差し替え対策)
  const _Game_Screen_showPicture = Game_Screen.prototype.showPicture;
  Game_Screen.prototype.showPicture = function(pictureId, name, origin, x, y, scaleX, scaleY, opacity, blendMode) {
    _Game_Screen_showPicture.call(this, pictureId, name, origin, x, y, scaleX, scaleY, opacity, blendMode);
    const store = ensureStore();
    if (!store) return;
    const pid = Number(pictureId);
    const ceid = Number(store[pid] || 0);
    if (pid > 0 && ceid > 0) applyBind(pid, ceid);
  };

  // 4) Scene_Map 開始時にも再適用 (ロード後の初期化タイミング差の吸収)
  const _Scene_Map_start = Scene_Map.prototype.start;
  Scene_Map.prototype.start = function() {
    _Scene_Map_start.call(this);
    applyAllExistingPictures();
  };
})();
